﻿using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Linq;
using System.Security;
using System.ServiceModel;
using System.Text;
using VA.PPMS.Context;
using VA.PPMS.CRM.Plugins.Helper;

namespace VA.PPMS.CRM.Plugins
{
    public class CareSiteCreate : IPlugin
    {
        private const string PluginName = "CareSiteCreate";
        private IOrganizationService _service;
        private ITracingService _tracingService;

        public void Execute(IServiceProvider serviceProvider)
        {
            // Tracing service for debugging
            _tracingService = (ITracingService)serviceProvider.GetService(typeof(ITracingService));

            // Get execution context
            IPluginExecutionContext context = (IPluginExecutionContext)serviceProvider.GetService(typeof(IPluginExecutionContext));

            if (context.InputParameters.Contains("Target") && context.InputParameters["Target"] is Entity)
            {
                _tracingService.Trace("Begin");

                // Obtain the target entity from the input parameters.
                Entity entity = (Entity)context.InputParameters["Target"];

                // Verify target entity type
                if (entity.LogicalName != "ppms_caresite")
                    return;

                _tracingService.Trace("Entity found");

                // Get organization service reference
                IOrganizationServiceFactory serviceFactory = (IOrganizationServiceFactory)serviceProvider.GetService(typeof(IOrganizationServiceFactory));
                _service = serviceFactory.CreateOrganizationService(context.UserId);

                // Verify service reference exists
                if (_service == null)
                {
                    _tracingService.Trace("Service is unavailable.");
                    return;
                }

                try
                {
                    using (var svc = new PpmsContext(_service))
                    {
                        _tracingService.Trace("Getting address attributes for care site...");
                        // Get Care Site Address Attributes
                        var careSite = svc.ppms_caresiteSet.First(i => i.Id == entity.Id);

                        _tracingService.Trace("Processing Care Site Address: {0}", careSite.ppms_name);

                        // Check to see if address is provided
                        if (careSite != null)
                        {
                            string streetAddress = careSite.ppms_address_line1;
                            string city = careSite.ppms_address_city;
                            string state = careSite.ppms_statename;
                            string zip = careSite.ppms_address_postalcode;
                            string address = streetAddress + " " + city + " " + state + " " + zip;

                            //Check the Use Address Validation? Field
                            bool useAddressValidation = (bool)careSite.Attributes["ppms_useaddressvalidationservice"];

                            if (useAddressValidation == true)
                            {
                                _tracingService.Trace("Address Validation Service");
                                //Address & Geocode Validation
                                var addressValidationData = AddressValidationHelper.ValidateWcDWS(streetAddress, city, state, zip, _tracingService);


                                if (addressValidationData != null && addressValidationData.ConfidenceScore >= 95)
                                {
                                    _tracingService.Trace("Address Validation successful");
                                    // Set geocode values                                  
                                    var coordinates = new PpmsCoordinate()
                                    {
                                        ConfidenceScore = addressValidationData.ConfidenceScore,
                                        Latitude = addressValidationData.Latitude,
                                        Longitude = addressValidationData.Longitude
                                    };
                                    SetCoordinates(entity, coordinates);
                                    SetAddressValidated(entity);
                                    //Set Provider Validation for Parent Provider related to Care Site
                                    //UpdateParentProviderValidation(careSite);
                                    // Create provider validation entities
                                    _tracingService.Trace("Create provider validation entries...");
                                    SetProviderValidation(entity);
                                }
                                else
                                {
                                    DeactiveRelations(entity);
                                }
                            }
                            else
                            {
                                _tracingService.Trace("Geocode Only");
                                //Geocoding only Validation
                                var addressData = GeocodeHelper.Geocode(address);
                                if (addressData != null)
                                {
                                    _tracingService.Trace("Address Validation successful");
                                    // Set geocode values
                                    var coordinates = new PpmsCoordinate()
                                    {
                                        //ConfidenceScore = addressData.ConfidenceScore,
                                        Latitude = addressData.Latitude,
                                        Longitude = addressData.Longitude
                                    };
                                    SetCoordinates(entity, coordinates);
                                    SetGeocoded(entity);
                                    // Create provider validation entities
                                    _tracingService.Trace("Create provider validation entries...");
                                    SetProviderValidation(entity);
                                }
                                else
                                {
                                    DeactiveRelations(entity);
                                }
                            }
                        }                       
                    }
                }
                catch (FaultException<OrganizationServiceFault> ex)
                {
                    _tracingService.Trace("Fault: {0}", ex.ToString());
                    throw new InvalidPluginExecutionException(String.Format("An error occurred in {0}.", PluginName), ex);
                }
                catch (Exception ex)
                {
                    _tracingService.Trace("Exception: {0}", ex.ToString());
                    return;
                }
            }
            _tracingService.Trace("Done");
        }

        private void SetCoordinates(Entity entity, PpmsCoordinate coordinates)
        {
            // check if change needs to be made
            if (entity != null)
            {
                // set latitude
                entity.Attributes["ppms_address_latitude"] = coordinates.Latitude.ToString();
                // set longitude
                entity.Attributes["ppms_address_longitude"] = coordinates.Longitude.ToString();
                // save changes
                _service.Update(entity);
            }
        }
        private void SetGeocoded(Entity entity)
        {
            // check if change needs to be made
            if (entity != null)
            {
                //Set Entity Geocode and Address Validated fields to True
                entity.Attributes["ppms_geocoded"] = true;
                // save changes
                _service.Update(entity);
            }
        }

        private void SetAddressValidated(Entity entity)
        {
            // check if change needs to be made
            if (entity != null)
            {
                //Set Entity Geocode and Address Validated fields to True
                entity.Attributes["ppms_geocoded"] = true;
                entity.Attributes["ppms_addressvalidated"] = true;

                // save changes
                _service.Update(entity);
            }
        }

        private void DeactiveRelations(Entity entity)
        {

            _tracingService.Trace("Address Validation unsuccessful");

            //Address Validation unsuccessful, set state code
            _service.Execute(PpmsHelper.GetDeactivateRequest(entity,
                (int)PpmsHelper.CareSite_StatusCode.AddressValidationFailure));



            // Deactivate owning group
            var org = entity.GetAttributeValue<EntityReference>("ppms_organization");
            if (org != null)
            {
                _tracingService.Trace("Deactivate organization");
                var provider = GetProvider(org.Id);
                if (provider != null)
                {
                    _service.Execute(PpmsHelper.GetDeactivateRequest(provider,
                        (int)PpmsHelper.Account_StatusCode.AddressValidation));
                }
            }

            // Deactivate provider services
            var list = GetActiveServicesByCareSite(entity.Id);
            if (list != null)
            {
                _tracingService.Trace("Deactivate provider services");
                foreach (var item in list.Entities)
                {
                    _service.Execute(PpmsHelper.GetDeactivateRequest(item,
                        (int)PpmsHelper.ProviderService_StatusCode.AddressValidationFailure));
                }
            }



        }

        private void UpdateParentProviderValidation(ppms_caresite careSite)
        {
            using (var svc = new PpmsContext(_service))
            {
                if (careSite != null && careSite.ppms_organization != null)
                {
                    _tracingService.Trace("Update Parent Provider");
                    var prov = svc.AccountSet.FirstOrDefault(i => i.Id == careSite.ppms_organization.Id);
                    if (prov != null)
                    {
                        // set Geocoded Validation
                        prov.Attributes["ppms_geocoded"] = true;
                        // set Address Validation
                        prov.Attributes["ppms_addressvalidated"] = true;
                        // save changes
                        _service.Update(prov);
                    }
                }
            }
        }

        private Entity GetProvider(Guid providerId)
        {
            return _service.Retrieve("account", providerId, new ColumnSet(new string[] { "accountid", "name" }));
        }

        private EntityCollection GetActiveServicesByCareSite(Guid careSiteId)
        {
            FilterExpression filter = new FilterExpression();
            filter.AddCondition("statecode", ConditionOperator.Equal, (int)PpmsHelper.AccountState.Active);
            filter.AddCondition("ppms_caresite", ConditionOperator.Equal, careSiteId);

            QueryExpression query = new QueryExpression("ppms_providerservice");
            query.ColumnSet.AddColumns("ppms_providerserviceid", "ppms_name", "ppms_providerid");
            query.Criteria.AddFilter(filter);

            return _service.RetrieveMultiple(query);
        }

        private string GetAddress(Entity entity)
        {
            var sb = new StringBuilder();
            if (entity != null)
            {
                AddAddressString(sb, entity, "ppms_address_line1");
                AddAddressString(sb, entity, "ppms_address_city");
                AddAddressEntity(sb, entity, "ppms_address_state");
                AddAddressString(sb, entity, "ppms_address_postalcode");
                AddAddressString(sb, entity, "ppms_address_county");
                AddAddressString(sb, entity, "ppms_address_country");
            }

            return sb.ToString();
        }

        private void AddAddressString(StringBuilder sb, Entity entity, string propertyName)
        {
            var element = entity.GetAttributeValue<string>(propertyName);
            if (!string.IsNullOrEmpty(element))
            {
                if (sb.Length > 0) sb.Append(",");
                sb.Append(element);
            }
        }

        private void AddAddressEntity(StringBuilder sb, Entity entity, string propertyName)
        {
            var element = entity.GetAttributeValue<EntityReference>(propertyName);
            if (element != null && !string.IsNullOrEmpty(element.Name))
            {
                if (sb.Length > 0) sb.Append(",");
                sb.Append(element.Name);
            }
        }

        private void SetProviderValidation(Entity entity)
        {
            // Get associated provider services
            var services = GetActiveServicesByCareSite(entity.Id);
            if (services != null && services.Entities.Count > 0)
            {
                foreach (var service in services.Entities)
                {
                    // Create provider validations
                    AddProviderValidations(service.GetAttributeValue<EntityReference>("ppms_providerid"));
                }
            }
        }

        private bool AddProviderValidations(EntityReference provider)
        {
            if (provider != null)
            {
                // Add Address validation
                PpmsHelper.AddProviderValidation(_service, provider, PpmsHelper.Validation_Type.Address);

                // Add Geocode validation
                PpmsHelper.AddProviderValidation(_service, provider, PpmsHelper.Validation_Type.Geocode);

                return true;
            }

            return false;
        }
    }
}
